// Common and shared funtions
var express = require('express');
var http = require('util');
var errorHandler = require('./error');
var errorTransformer = require('./error_transformer');
var thrift = require('thrift');
var WIPService = require('../gen-nodejs/WIPService');
var thriftPool = require('node-thrift-pool');
var debug = require('debug')('api');
var basicAuth = require('basic-auth');
var path = require('path');
var fs = require('fs');
var xml2js = require('xml2js');

// detect cloud version:
// 0 - on premise
// 1 - cloud 1.x
// 2 - cloud 2.x
var cloudVersion = function() {
	if (process.platform === "linux") {
		var filePath = __dirname + "/../buildVersionCloud.json";
		filePath = path.resolve(filePath);
		var contents = fs.readFileSync(filePath);
		var buildVersion = JSON.parse(contents);
		if (buildVersion.buildVersion.startsWith('1.')) {
			return 1;
		} else if (buildVersion.buildVersion.startsWith('2.')) {
			return 2;
		} else {
			return 0; // return something else like 3?
		}
	} else {
		return 0;
	}
}

var validateRequest = function(req, res, allowedMethods){
	if(allowedMethods && -1 === allowedMethods.indexOf(req.method.toUpperCase())){
		errorHandler.handleError("MethodNotAllowed", req, res, allowedMethods.join(','));
	}
	// if(!req.get('FM-Data-token')){
	// 	errorHandler.handleError("Unauthorized", req, res, 'Missing FM-Data-token.');
	// 	return false;
	// }
	//content-type must be application/json for POST and PUT
	if((req.method ==="POST" || req.method ==="PUT" || req.method ==="PATCH")  && !req.is('application/json')){
		errorHandler.handleError("UnsupportedMediaType", req, res);
		return false;
	}
	return true;
}

var validateUploadRequest = function(req, res){
	if(req.method.toUpperCase() !== "POST"){
		errorHandler.handleError("MethodNotAllowed", req, res, req.method.toUpperCase());
	}
	//content-type must be application/json for POST and PUT
	if(req.method ==="POST" && !req.is('multipart/form-data')){
		errorHandler.handleError("UnsupportedMediaType", req, res);
		return false;
	}
	return true;
}

var parseClientIp = function(req){
	var ipStr = req.get('X-Forwarded-For');  //bug 236856, get X-Forwarded-For first IP as public client IP
	var ips = [];
	if (!ipStr) {
		ipStr = "";
	} else {
		ips = ipStr.split(',');
		ipStr = ips[0];
	}
	return ipStr;
}

var parseTimeoutPref = function() {
        var defaultTimeout = 300; //default timeout to 300 seconds
	var timeout = defaultTimeout;
        var timeout_property="server/parameters/config/wip-timeout-secs";
        var filePath = __dirname + "/../../conf/wpe.prefs";
        filePath = path.resolve(filePath);
        var contents = fs.readFileSync(filePath).toString().split("\n");
        for(i in contents){
                if(contents[i].startsWith(timeout_property)) {
                        timeout = contents[i].split("=")[1];
                        break;
                }
        }

	timeout = parseInt(timeout);	

	if(typeof(timeout)==="number" && !isNaN(timeout)){
		if(timeout <= 0)
		{
			timeout = defaultTimeout;
		}

        	return timeout * 1000; //convert to milliseconds
	}else{
		
		return defaultTimeout * 1000;
	}
}

//with thriftPool
try{
	var thrift_client = thriftPool(thrift, WIPService,
								{
									host: "localhost",
									port: 8989,
									max_connections:30,
									min_connections:10
								},
								{
									debug: true,
									timeout: parseInt(parseTimeoutPref(), 10),
									idle_timeout: 3000000, //the default one is 3000 which is too short
									transport : thrift.TFramedTransport,
    									protocol : thrift.TBinaryProtocol
								});
}catch(err){
	debug(err);
}

var thriftExceptionHandler = function(err, req, res){
	debug(err)
	errorHandler.handleError('FMServiceError', req, res, err.message);
}

//handleThrifReturn
var handleThrifReturn = function(thrifError, thrifResult, req, res, callback) {
		if(handleThrifErr(thrifError, thrifResult, req, res)){
			res.set('Content-Type', 'application/json');
			res.set('Cache-Control', 'no-cache');
			if (typeof(callback) === 'function') {
				callback(thrifResult, res);
			}
			res.send(thrifResult);
		}
}

var handleThrifErr = function(thrifError, thrifResult, req, res){
		 // Thrift will return thrifError===null if success, otherwise it set a string of error message
		 if (thrifError !== null) {
			 debug('Thrift Error', thrifError);
			 errorHandler.handleError('InternalServerError', req, res, thrifError.message ? thrifError.message : thrifError);
			 return false;
		 } else {
			 //try parsing the returned thrifResult and catch if it has Invalid json format
			 try {
				 var result = JSON.parse(thrifResult);
			 } catch(e) {
				 errorHandler.handleError('InternalServerError', req, res, "Invalid json string returned from internal server.");
				 return false;
			 }
			 // look at first message
			 errorMessage = result.messages[0];
			 //Backend service will return errorCode:'0' if service success, otherwise it will be a none 0 error code.
			 if (errorMessage.code !== '0') {
				 debug('FM Service error ', result);
				 if (errorMessage.code === '952') {//watch request with invalid token
					 watchIt(req);
				 }
				 //transform internal error to proper error to client properly
				 var error = errorTransformer.transform(errorMessage);
				 errorHandler.handleError(error.type, req, res, errorMessage, errorMessage.code, result.response);
				 if (result.hasOwnProperty('response')) {
					 errorMessage.response = result.response;
				 }
				 return false;
			 } else {
				 return true;
			 }
		 }

};

var watchList = new Map();  //ip : [1470955745530,1470955746878,..,1470955749196]
var blackList = new Map();	//ip : true;
const allowAttempts = 100;			//blackList the ip for 10 minutes if it requests more than allowAttempts with Invalid token withing allowInterval.
const allowInterval = 10000;
const blackListTimeout = 10*60*1000;
var isBlackListed = function(req){
		return blackList.has(req.ip);
};
var watchIt = function(req){
		var time = (new Date()).getTime();
		var access = watchList.get(req.ip);

		if(access === undefined){
			watchList.set(req.ip, [time]);
		}else{
			while((time - access[0]) > allowInterval){ //remove all logs older than allowInterval
					access.shift();
			}
			var attempts = access.push(time);
			//debug(req.ip, access);
			watchList.set(req.ip, access);
			if(attempts === allowAttempts){  //if attempts more than allowed, blackList it unitil timeout
				watchList.delete(req.ip);
				blackList.set(req.ip, 'true');
				setTimeout(function(){blackList.delete(req.ip)}, blackListTimeout);
				return true;
			}

		}
};

//check if the request is from Tableau Connector
var isFromTableau = function(req){
	return (req.get('user-agent').indexOf('tabprotosrv')>-1
	|| req.get('user-agent').indexOf('Qt/')>-1
	|| req.get('user-agent').indexOf('Tableau')>-1
	|| req.get('user-agent').indexOf('AppleWebKit/538.1')>-1); //For Tableau 10
}

var parseParams = function(listOfAllowedParams, inParams, outParams) {
	for (var param in inParams) {
		if (listOfAllowedParams.indexOf(param) != -1) {
			outParams[param] = inParams[param];
			delete inParams[param];
		}
	}
}

var setAuthentication = function(req, res, params, isLogin) {
	var toReturn = true;
	if (req.swagger.params.hasOwnProperty('Authorization') && req.swagger.params.Authorization.value) {
		var authorization = req.swagger.params.Authorization.value;
		var auth = basicAuth(req);
		if (!auth) {
			if (cloudVersion() === 2) {
				if (authorization.toLowerCase().startsWith("bearer ")) {
					if (isLogin) {
						errorHandler.handleError('Unauthorized', req, res, "Authorization header 'Bearer' invalid.","212");
						toReturn = false;
					} else {
						params.token = authorization.split(" ")[1];
					}
				} else if (authorization.toLowerCase().startsWith("fmid ")) {
					if (isLogin) {
						params.fmidToken = authorization.split(" ")[1];
					} else {
						errorHandler.handleError('Unauthorized', req, res, "Authorization header 'Fmid' invalid.","212");
						toReturn = false;
					}
				} else {
					errorHandler.handleError('Unauthorized', req, res, "Authorization header missing 'Fmid' or 'Bearer'.","10");
					toReturn = false;
				}
			} else {
				if (authorization.toLowerCase().startsWith("bearer ")) {
					params.token = authorization.split(" ")[1];
				} else if (authorization.toLowerCase().startsWith("basic ")) {
					errorHandler.handleError('Unauthorized', req, res, "Authorization header 'Basic' invalid.","212");	
					toReturn = false;	
				} else {
					errorHandler.handleError('Unauthorized', req, res, "Authorization header missing 'Basic' or 'Bearer'.","10");
					toReturn = false;
				}
			}
		} else {
			if (cloudVersion() === 2) {
				errorHandler.handleError('Unauthorized', req, res, "Authorization header 'Basic' invalid.","212");
				toReturn = false;
			} else {
				if (isLogin) {
					params.username = auth.name;
					params.password = auth.pass;
				} else {
					errorHandler.handleError('Unauthorized', req, res, "Authorization header 'Basic' invalid.","212");
					toReturn = false;
				}
			}
		}
	} else if ((req.swagger.params.hasOwnProperty('X-FM-Data-OAuth-Request-Id') && req.swagger.params['X-FM-Data-OAuth-Request-Id'].value) &&
			   (req.swagger.params.hasOwnProperty('X-FM-Data-OAuth-Identifier') && req.swagger.params['X-FM-Data-OAuth-Identifier'].value)) {
		params.oAuthRequestId = req.swagger.params['X-FM-Data-OAuth-Request-Id'].value;
		params.oAuthIdentifier = req.swagger.params['X-FM-Data-OAuth-Identifier'].value;
	} else {
		errorHandler.handleError('Unauthorized', req, res, "HTTP Authorization header or OAuth headers are missing.","10");
		toReturn = false;
	}

	return toReturn;
}

// Return value of premises in Admin/conf/deployment.xml
//     0 - on-premises
// non 0 - cloud
var checkPremises = function() {
	var premises = '0';
	var file = '/../../../../Admin/conf/deployment.xml';
    var filePath = __dirname + file;
    filePath = path.resolve(filePath);

    var deploymentInfo = fs.readFileSync(filePath);
    xml2js.parseString(deploymentInfo, {async: false}, function(err, result) {
		if (err) {
			console.log('Error parsing deployment.xml.');
			return;
		}
		let tmp = result['fm-service-configuration'].host[0]['$'].premises;
		if (typeof(tmp) !== 'undefined') {
			premises = tmp;
		}
    });

	return premises;
}

module.exports.validateRequest = validateRequest;
module.exports.validateUploadRequest = validateUploadRequest;
module.exports.parseClientIp = parseClientIp;
module.exports.thrift_client = thrift_client;
module.exports.thriftExceptionHandler = thriftExceptionHandler;
module.exports.handleThrifReturn = handleThrifReturn;
module.exports.isBlackListed = isBlackListed;
module.exports.isFromTableau = isFromTableau;
module.exports.parseParams = parseParams;
module.exports.setAuthentication = setAuthentication;
module.exports.cloudVersion = cloudVersion;
module.exports.parseTimeoutPref = parseTimeoutPref;
module.exports.checkPremises = checkPremises;
